/*
 * Copyright (c) Cloud Software Group, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *   1) Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *
 *   2) Redistributions in binary form must reproduce the above
 *      copyright notice, this list of conditions and the following
 *      disclaimer in the documentation and/or other materials
 *      provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */


using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Globalization;
using System.Linq;
using Newtonsoft.Json;


namespace XenAPI
{
    /// <summary>
    /// A virtual network interface
    /// First published in XenServer 4.0.
    /// </summary>
    public partial class VIF : XenObject<VIF>
    {
        #region Constructors

        public VIF()
        {
        }

        public VIF(string uuid,
            List<vif_operations> allowed_operations,
            Dictionary<string, vif_operations> current_operations,
            string device,
            XenRef<Network> network,
            XenRef<VM> VM,
            string MAC,
            long MTU,
            Dictionary<string, string> other_config,
            bool currently_attached,
            long status_code,
            string status_detail,
            Dictionary<string, string> runtime_properties,
            string qos_algorithm_type,
            Dictionary<string, string> qos_algorithm_params,
            string[] qos_supported_algorithms,
            XenRef<VIF_metrics> metrics,
            bool MAC_autogenerated,
            vif_locking_mode locking_mode,
            string[] ipv4_allowed,
            string[] ipv6_allowed,
            vif_ipv4_configuration_mode ipv4_configuration_mode,
            string[] ipv4_addresses,
            string ipv4_gateway,
            vif_ipv6_configuration_mode ipv6_configuration_mode,
            string[] ipv6_addresses,
            string ipv6_gateway)
        {
            this.uuid = uuid;
            this.allowed_operations = allowed_operations;
            this.current_operations = current_operations;
            this.device = device;
            this.network = network;
            this.VM = VM;
            this.MAC = MAC;
            this.MTU = MTU;
            this.other_config = other_config;
            this.currently_attached = currently_attached;
            this.status_code = status_code;
            this.status_detail = status_detail;
            this.runtime_properties = runtime_properties;
            this.qos_algorithm_type = qos_algorithm_type;
            this.qos_algorithm_params = qos_algorithm_params;
            this.qos_supported_algorithms = qos_supported_algorithms;
            this.metrics = metrics;
            this.MAC_autogenerated = MAC_autogenerated;
            this.locking_mode = locking_mode;
            this.ipv4_allowed = ipv4_allowed;
            this.ipv6_allowed = ipv6_allowed;
            this.ipv4_configuration_mode = ipv4_configuration_mode;
            this.ipv4_addresses = ipv4_addresses;
            this.ipv4_gateway = ipv4_gateway;
            this.ipv6_configuration_mode = ipv6_configuration_mode;
            this.ipv6_addresses = ipv6_addresses;
            this.ipv6_gateway = ipv6_gateway;
        }

        /// <summary>
        /// Creates a new VIF from a Hashtable.
        /// Note that the fields not contained in the Hashtable
        /// will be created with their default values.
        /// </summary>
        /// <param name="table"></param>
        public VIF(Hashtable table)
            : this()
        {
            UpdateFrom(table);
        }

        #endregion

        /// <summary>
        /// Updates each field of this instance with the value of
        /// the corresponding field of a given VIF.
        /// </summary>
        public override void UpdateFrom(VIF record)
        {
            uuid = record.uuid;
            allowed_operations = record.allowed_operations;
            current_operations = record.current_operations;
            device = record.device;
            network = record.network;
            VM = record.VM;
            MAC = record.MAC;
            MTU = record.MTU;
            other_config = record.other_config;
            currently_attached = record.currently_attached;
            status_code = record.status_code;
            status_detail = record.status_detail;
            runtime_properties = record.runtime_properties;
            qos_algorithm_type = record.qos_algorithm_type;
            qos_algorithm_params = record.qos_algorithm_params;
            qos_supported_algorithms = record.qos_supported_algorithms;
            metrics = record.metrics;
            MAC_autogenerated = record.MAC_autogenerated;
            locking_mode = record.locking_mode;
            ipv4_allowed = record.ipv4_allowed;
            ipv6_allowed = record.ipv6_allowed;
            ipv4_configuration_mode = record.ipv4_configuration_mode;
            ipv4_addresses = record.ipv4_addresses;
            ipv4_gateway = record.ipv4_gateway;
            ipv6_configuration_mode = record.ipv6_configuration_mode;
            ipv6_addresses = record.ipv6_addresses;
            ipv6_gateway = record.ipv6_gateway;
        }

        /// <summary>
        /// Given a Hashtable with field-value pairs, it updates the fields of this VIF
        /// with the values listed in the Hashtable. Note that only the fields contained
        /// in the Hashtable will be updated and the rest will remain the same.
        /// </summary>
        /// <param name="table"></param>
        public void UpdateFrom(Hashtable table)
        {
            if (table.ContainsKey("uuid"))
                uuid = Marshalling.ParseString(table, "uuid");
            if (table.ContainsKey("allowed_operations"))
                allowed_operations = Helper.StringArrayToEnumList<vif_operations>(Marshalling.ParseStringArray(table, "allowed_operations"));
            if (table.ContainsKey("current_operations"))
                current_operations = Maps.convert_from_proxy_string_vif_operations(Marshalling.ParseHashTable(table, "current_operations"));
            if (table.ContainsKey("device"))
                device = Marshalling.ParseString(table, "device");
            if (table.ContainsKey("network"))
                network = Marshalling.ParseRef<Network>(table, "network");
            if (table.ContainsKey("VM"))
                VM = Marshalling.ParseRef<VM>(table, "VM");
            if (table.ContainsKey("MAC"))
                MAC = Marshalling.ParseString(table, "MAC");
            if (table.ContainsKey("MTU"))
                MTU = Marshalling.ParseLong(table, "MTU");
            if (table.ContainsKey("other_config"))
                other_config = Maps.convert_from_proxy_string_string(Marshalling.ParseHashTable(table, "other_config"));
            if (table.ContainsKey("currently_attached"))
                currently_attached = Marshalling.ParseBool(table, "currently_attached");
            if (table.ContainsKey("status_code"))
                status_code = Marshalling.ParseLong(table, "status_code");
            if (table.ContainsKey("status_detail"))
                status_detail = Marshalling.ParseString(table, "status_detail");
            if (table.ContainsKey("runtime_properties"))
                runtime_properties = Maps.convert_from_proxy_string_string(Marshalling.ParseHashTable(table, "runtime_properties"));
            if (table.ContainsKey("qos_algorithm_type"))
                qos_algorithm_type = Marshalling.ParseString(table, "qos_algorithm_type");
            if (table.ContainsKey("qos_algorithm_params"))
                qos_algorithm_params = Maps.convert_from_proxy_string_string(Marshalling.ParseHashTable(table, "qos_algorithm_params"));
            if (table.ContainsKey("qos_supported_algorithms"))
                qos_supported_algorithms = Marshalling.ParseStringArray(table, "qos_supported_algorithms");
            if (table.ContainsKey("metrics"))
                metrics = Marshalling.ParseRef<VIF_metrics>(table, "metrics");
            if (table.ContainsKey("MAC_autogenerated"))
                MAC_autogenerated = Marshalling.ParseBool(table, "MAC_autogenerated");
            if (table.ContainsKey("locking_mode"))
                locking_mode = (vif_locking_mode)Helper.EnumParseDefault(typeof(vif_locking_mode), Marshalling.ParseString(table, "locking_mode"));
            if (table.ContainsKey("ipv4_allowed"))
                ipv4_allowed = Marshalling.ParseStringArray(table, "ipv4_allowed");
            if (table.ContainsKey("ipv6_allowed"))
                ipv6_allowed = Marshalling.ParseStringArray(table, "ipv6_allowed");
            if (table.ContainsKey("ipv4_configuration_mode"))
                ipv4_configuration_mode = (vif_ipv4_configuration_mode)Helper.EnumParseDefault(typeof(vif_ipv4_configuration_mode), Marshalling.ParseString(table, "ipv4_configuration_mode"));
            if (table.ContainsKey("ipv4_addresses"))
                ipv4_addresses = Marshalling.ParseStringArray(table, "ipv4_addresses");
            if (table.ContainsKey("ipv4_gateway"))
                ipv4_gateway = Marshalling.ParseString(table, "ipv4_gateway");
            if (table.ContainsKey("ipv6_configuration_mode"))
                ipv6_configuration_mode = (vif_ipv6_configuration_mode)Helper.EnumParseDefault(typeof(vif_ipv6_configuration_mode), Marshalling.ParseString(table, "ipv6_configuration_mode"));
            if (table.ContainsKey("ipv6_addresses"))
                ipv6_addresses = Marshalling.ParseStringArray(table, "ipv6_addresses");
            if (table.ContainsKey("ipv6_gateway"))
                ipv6_gateway = Marshalling.ParseString(table, "ipv6_gateway");
        }

        public bool DeepEquals(VIF other, bool ignoreCurrentOperations)
        {
            if (ReferenceEquals(null, other))
                return false;
            if (ReferenceEquals(this, other))
                return true;

            if (!ignoreCurrentOperations && !Helper.AreEqual2(this.current_operations, other.current_operations))
                return false;

            return Helper.AreEqual2(this._uuid, other._uuid) &&
                Helper.AreEqual2(this._allowed_operations, other._allowed_operations) &&
                Helper.AreEqual2(this._device, other._device) &&
                Helper.AreEqual2(this._network, other._network) &&
                Helper.AreEqual2(this._VM, other._VM) &&
                Helper.AreEqual2(this._MAC, other._MAC) &&
                Helper.AreEqual2(this._MTU, other._MTU) &&
                Helper.AreEqual2(this._other_config, other._other_config) &&
                Helper.AreEqual2(this._currently_attached, other._currently_attached) &&
                Helper.AreEqual2(this._status_code, other._status_code) &&
                Helper.AreEqual2(this._status_detail, other._status_detail) &&
                Helper.AreEqual2(this._runtime_properties, other._runtime_properties) &&
                Helper.AreEqual2(this._qos_algorithm_type, other._qos_algorithm_type) &&
                Helper.AreEqual2(this._qos_algorithm_params, other._qos_algorithm_params) &&
                Helper.AreEqual2(this._qos_supported_algorithms, other._qos_supported_algorithms) &&
                Helper.AreEqual2(this._metrics, other._metrics) &&
                Helper.AreEqual2(this._MAC_autogenerated, other._MAC_autogenerated) &&
                Helper.AreEqual2(this._locking_mode, other._locking_mode) &&
                Helper.AreEqual2(this._ipv4_allowed, other._ipv4_allowed) &&
                Helper.AreEqual2(this._ipv6_allowed, other._ipv6_allowed) &&
                Helper.AreEqual2(this._ipv4_configuration_mode, other._ipv4_configuration_mode) &&
                Helper.AreEqual2(this._ipv4_addresses, other._ipv4_addresses) &&
                Helper.AreEqual2(this._ipv4_gateway, other._ipv4_gateway) &&
                Helper.AreEqual2(this._ipv6_configuration_mode, other._ipv6_configuration_mode) &&
                Helper.AreEqual2(this._ipv6_addresses, other._ipv6_addresses) &&
                Helper.AreEqual2(this._ipv6_gateway, other._ipv6_gateway);
        }

        public override string SaveChanges(Session session, string opaqueRef, VIF server)
        {
            if (opaqueRef == null)
            {
                var reference = create(session, this);
                return reference == null ? null : reference.opaque_ref;
            }
            else
            {
                if (!Helper.AreEqual2(_other_config, server._other_config))
                {
                    VIF.set_other_config(session, opaqueRef, _other_config);
                }
                if (!Helper.AreEqual2(_qos_algorithm_type, server._qos_algorithm_type))
                {
                    VIF.set_qos_algorithm_type(session, opaqueRef, _qos_algorithm_type);
                }
                if (!Helper.AreEqual2(_qos_algorithm_params, server._qos_algorithm_params))
                {
                    VIF.set_qos_algorithm_params(session, opaqueRef, _qos_algorithm_params);
                }
                if (!Helper.AreEqual2(_locking_mode, server._locking_mode))
                {
                    VIF.set_locking_mode(session, opaqueRef, _locking_mode);
                }
                if (!Helper.AreEqual2(_ipv4_allowed, server._ipv4_allowed))
                {
                    VIF.set_ipv4_allowed(session, opaqueRef, _ipv4_allowed);
                }
                if (!Helper.AreEqual2(_ipv6_allowed, server._ipv6_allowed))
                {
                    VIF.set_ipv6_allowed(session, opaqueRef, _ipv6_allowed);
                }

                return null;
            }
        }

        /// <summary>
        /// Get a record containing the current state of the given VIF.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static VIF get_record(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_record(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get a reference to the VIF instance with the specified UUID.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_uuid">UUID of object to return</param>
        public static XenRef<VIF> get_by_uuid(Session session, string _uuid)
        {
            return session.JsonRpcClient.vif_get_by_uuid(session.opaque_ref, _uuid);
        }

        /// <summary>
        /// Create a new VIF instance, and return its handle.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_record">All constructor arguments</param>
        public static XenRef<VIF> create(Session session, VIF _record)
        {
            return session.JsonRpcClient.vif_create(session.opaque_ref, _record);
        }

        /// <summary>
        /// Create a new VIF instance, and return its handle.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_record">All constructor arguments</param>
        public static XenRef<Task> async_create(Session session, VIF _record)
        {
          return session.JsonRpcClient.async_vif_create(session.opaque_ref, _record);
        }

        /// <summary>
        /// Destroy the specified VIF instance.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static void destroy(Session session, string _vif)
        {
            session.JsonRpcClient.vif_destroy(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Destroy the specified VIF instance.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static XenRef<Task> async_destroy(Session session, string _vif)
        {
          return session.JsonRpcClient.async_vif_destroy(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the uuid field of the given VIF.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static string get_uuid(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_uuid(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the allowed_operations field of the given VIF.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static List<vif_operations> get_allowed_operations(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_allowed_operations(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the current_operations field of the given VIF.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static Dictionary<string, vif_operations> get_current_operations(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_current_operations(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the device field of the given VIF.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static string get_device(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_device(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the network field of the given VIF.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static XenRef<Network> get_network(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_network(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the VM field of the given VIF.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static XenRef<VM> get_VM(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_vm(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the MAC field of the given VIF.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static string get_MAC(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_mac(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the MTU field of the given VIF.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static long get_MTU(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_mtu(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the other_config field of the given VIF.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static Dictionary<string, string> get_other_config(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_other_config(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the currently_attached field of the given VIF.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static bool get_currently_attached(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_currently_attached(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the status_code field of the given VIF.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static long get_status_code(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_status_code(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the status_detail field of the given VIF.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static string get_status_detail(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_status_detail(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the runtime_properties field of the given VIF.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static Dictionary<string, string> get_runtime_properties(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_runtime_properties(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the qos/algorithm_type field of the given VIF.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static string get_qos_algorithm_type(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_qos_algorithm_type(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the qos/algorithm_params field of the given VIF.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static Dictionary<string, string> get_qos_algorithm_params(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_qos_algorithm_params(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the qos/supported_algorithms field of the given VIF.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static string[] get_qos_supported_algorithms(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_qos_supported_algorithms(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the metrics field of the given VIF.
        /// First published in XenServer 4.0.
        /// Deprecated since XenServer 6.1.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        [Deprecated("XenServer 6.1")]
        public static XenRef<VIF_metrics> get_metrics(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_metrics(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the MAC_autogenerated field of the given VIF.
        /// First published in XenServer 5.5.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static bool get_MAC_autogenerated(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_mac_autogenerated(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the locking_mode field of the given VIF.
        /// First published in XenServer 6.1.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static vif_locking_mode get_locking_mode(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_locking_mode(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the ipv4_allowed field of the given VIF.
        /// First published in XenServer 6.1.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static string[] get_ipv4_allowed(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_ipv4_allowed(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the ipv6_allowed field of the given VIF.
        /// First published in XenServer 6.1.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static string[] get_ipv6_allowed(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_ipv6_allowed(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the ipv4_configuration_mode field of the given VIF.
        /// First published in XenServer 7.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static vif_ipv4_configuration_mode get_ipv4_configuration_mode(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_ipv4_configuration_mode(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the ipv4_addresses field of the given VIF.
        /// First published in XenServer 7.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static string[] get_ipv4_addresses(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_ipv4_addresses(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the ipv4_gateway field of the given VIF.
        /// First published in XenServer 7.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static string get_ipv4_gateway(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_ipv4_gateway(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the ipv6_configuration_mode field of the given VIF.
        /// First published in XenServer 7.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static vif_ipv6_configuration_mode get_ipv6_configuration_mode(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_ipv6_configuration_mode(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the ipv6_addresses field of the given VIF.
        /// First published in XenServer 7.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static string[] get_ipv6_addresses(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_ipv6_addresses(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Get the ipv6_gateway field of the given VIF.
        /// First published in XenServer 7.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static string get_ipv6_gateway(Session session, string _vif)
        {
            return session.JsonRpcClient.vif_get_ipv6_gateway(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Set the other_config field of the given VIF.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_other_config">New value to set</param>
        public static void set_other_config(Session session, string _vif, Dictionary<string, string> _other_config)
        {
            session.JsonRpcClient.vif_set_other_config(session.opaque_ref, _vif, _other_config);
        }

        /// <summary>
        /// Add the given key-value pair to the other_config field of the given VIF.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_key">Key to add</param>
        /// <param name="_value">Value to add</param>
        public static void add_to_other_config(Session session, string _vif, string _key, string _value)
        {
            session.JsonRpcClient.vif_add_to_other_config(session.opaque_ref, _vif, _key, _value);
        }

        /// <summary>
        /// Remove the given key and its corresponding value from the other_config field of the given VIF.  If the key is not in that Map, then do nothing.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_key">Key to remove</param>
        public static void remove_from_other_config(Session session, string _vif, string _key)
        {
            session.JsonRpcClient.vif_remove_from_other_config(session.opaque_ref, _vif, _key);
        }

        /// <summary>
        /// Set the qos/algorithm_type field of the given VIF.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_algorithm_type">New value to set</param>
        public static void set_qos_algorithm_type(Session session, string _vif, string _algorithm_type)
        {
            session.JsonRpcClient.vif_set_qos_algorithm_type(session.opaque_ref, _vif, _algorithm_type);
        }

        /// <summary>
        /// Set the qos/algorithm_params field of the given VIF.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_algorithm_params">New value to set</param>
        public static void set_qos_algorithm_params(Session session, string _vif, Dictionary<string, string> _algorithm_params)
        {
            session.JsonRpcClient.vif_set_qos_algorithm_params(session.opaque_ref, _vif, _algorithm_params);
        }

        /// <summary>
        /// Add the given key-value pair to the qos/algorithm_params field of the given VIF.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_key">Key to add</param>
        /// <param name="_value">Value to add</param>
        public static void add_to_qos_algorithm_params(Session session, string _vif, string _key, string _value)
        {
            session.JsonRpcClient.vif_add_to_qos_algorithm_params(session.opaque_ref, _vif, _key, _value);
        }

        /// <summary>
        /// Remove the given key and its corresponding value from the qos/algorithm_params field of the given VIF.  If the key is not in that Map, then do nothing.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_key">Key to remove</param>
        public static void remove_from_qos_algorithm_params(Session session, string _vif, string _key)
        {
            session.JsonRpcClient.vif_remove_from_qos_algorithm_params(session.opaque_ref, _vif, _key);
        }

        /// <summary>
        /// Hotplug the specified VIF, dynamically attaching it to the running VM
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static void plug(Session session, string _vif)
        {
            session.JsonRpcClient.vif_plug(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Hotplug the specified VIF, dynamically attaching it to the running VM
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static XenRef<Task> async_plug(Session session, string _vif)
        {
          return session.JsonRpcClient.async_vif_plug(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Hot-unplug the specified VIF, dynamically unattaching it from the running VM
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static void unplug(Session session, string _vif)
        {
            session.JsonRpcClient.vif_unplug(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Hot-unplug the specified VIF, dynamically unattaching it from the running VM
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static XenRef<Task> async_unplug(Session session, string _vif)
        {
          return session.JsonRpcClient.async_vif_unplug(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Forcibly unplug the specified VIF
        /// First published in XenServer 6.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static void unplug_force(Session session, string _vif)
        {
            session.JsonRpcClient.vif_unplug_force(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Forcibly unplug the specified VIF
        /// First published in XenServer 6.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        public static XenRef<Task> async_unplug_force(Session session, string _vif)
        {
          return session.JsonRpcClient.async_vif_unplug_force(session.opaque_ref, _vif);
        }

        /// <summary>
        /// Move the specified VIF to the specified network, even while the VM is running
        /// First published in XenServer 7.1.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_network">The network to move it to</param>
        public static void move(Session session, string _vif, string _network)
        {
            session.JsonRpcClient.vif_move(session.opaque_ref, _vif, _network);
        }

        /// <summary>
        /// Move the specified VIF to the specified network, even while the VM is running
        /// First published in XenServer 7.1.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_network">The network to move it to</param>
        public static XenRef<Task> async_move(Session session, string _vif, string _network)
        {
          return session.JsonRpcClient.async_vif_move(session.opaque_ref, _vif, _network);
        }

        /// <summary>
        /// Set the locking mode for this VIF
        /// First published in XenServer 6.1.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_value">The new locking mode for the VIF</param>
        public static void set_locking_mode(Session session, string _vif, vif_locking_mode _value)
        {
            session.JsonRpcClient.vif_set_locking_mode(session.opaque_ref, _vif, _value);
        }

        /// <summary>
        /// Set the locking mode for this VIF
        /// First published in XenServer 6.1.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_value">The new locking mode for the VIF</param>
        public static XenRef<Task> async_set_locking_mode(Session session, string _vif, vif_locking_mode _value)
        {
          return session.JsonRpcClient.async_vif_set_locking_mode(session.opaque_ref, _vif, _value);
        }

        /// <summary>
        /// Set the IPv4 addresses to which traffic on this VIF can be restricted
        /// First published in XenServer 6.1.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_value">The IP addresses which will be associated with the VIF</param>
        public static void set_ipv4_allowed(Session session, string _vif, string[] _value)
        {
            session.JsonRpcClient.vif_set_ipv4_allowed(session.opaque_ref, _vif, _value);
        }

        /// <summary>
        /// Set the IPv4 addresses to which traffic on this VIF can be restricted
        /// First published in XenServer 6.1.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_value">The IP addresses which will be associated with the VIF</param>
        public static XenRef<Task> async_set_ipv4_allowed(Session session, string _vif, string[] _value)
        {
          return session.JsonRpcClient.async_vif_set_ipv4_allowed(session.opaque_ref, _vif, _value);
        }

        /// <summary>
        /// Associates an IPv4 address with this VIF
        /// First published in XenServer 6.1.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_value">The IP address which will be associated with the VIF</param>
        public static void add_ipv4_allowed(Session session, string _vif, string _value)
        {
            session.JsonRpcClient.vif_add_ipv4_allowed(session.opaque_ref, _vif, _value);
        }

        /// <summary>
        /// Associates an IPv4 address with this VIF
        /// First published in XenServer 6.1.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_value">The IP address which will be associated with the VIF</param>
        public static XenRef<Task> async_add_ipv4_allowed(Session session, string _vif, string _value)
        {
          return session.JsonRpcClient.async_vif_add_ipv4_allowed(session.opaque_ref, _vif, _value);
        }

        /// <summary>
        /// Removes an IPv4 address from this VIF
        /// First published in XenServer 6.1.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_value">The IP address which will be removed from the VIF</param>
        public static void remove_ipv4_allowed(Session session, string _vif, string _value)
        {
            session.JsonRpcClient.vif_remove_ipv4_allowed(session.opaque_ref, _vif, _value);
        }

        /// <summary>
        /// Removes an IPv4 address from this VIF
        /// First published in XenServer 6.1.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_value">The IP address which will be removed from the VIF</param>
        public static XenRef<Task> async_remove_ipv4_allowed(Session session, string _vif, string _value)
        {
          return session.JsonRpcClient.async_vif_remove_ipv4_allowed(session.opaque_ref, _vif, _value);
        }

        /// <summary>
        /// Set the IPv6 addresses to which traffic on this VIF can be restricted
        /// First published in XenServer 6.1.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_value">The IP addresses which will be associated with the VIF</param>
        public static void set_ipv6_allowed(Session session, string _vif, string[] _value)
        {
            session.JsonRpcClient.vif_set_ipv6_allowed(session.opaque_ref, _vif, _value);
        }

        /// <summary>
        /// Set the IPv6 addresses to which traffic on this VIF can be restricted
        /// First published in XenServer 6.1.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_value">The IP addresses which will be associated with the VIF</param>
        public static XenRef<Task> async_set_ipv6_allowed(Session session, string _vif, string[] _value)
        {
          return session.JsonRpcClient.async_vif_set_ipv6_allowed(session.opaque_ref, _vif, _value);
        }

        /// <summary>
        /// Associates an IPv6 address with this VIF
        /// First published in XenServer 6.1.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_value">The IP address which will be associated with the VIF</param>
        public static void add_ipv6_allowed(Session session, string _vif, string _value)
        {
            session.JsonRpcClient.vif_add_ipv6_allowed(session.opaque_ref, _vif, _value);
        }

        /// <summary>
        /// Associates an IPv6 address with this VIF
        /// First published in XenServer 6.1.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_value">The IP address which will be associated with the VIF</param>
        public static XenRef<Task> async_add_ipv6_allowed(Session session, string _vif, string _value)
        {
          return session.JsonRpcClient.async_vif_add_ipv6_allowed(session.opaque_ref, _vif, _value);
        }

        /// <summary>
        /// Removes an IPv6 address from this VIF
        /// First published in XenServer 6.1.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_value">The IP address which will be removed from the VIF</param>
        public static void remove_ipv6_allowed(Session session, string _vif, string _value)
        {
            session.JsonRpcClient.vif_remove_ipv6_allowed(session.opaque_ref, _vif, _value);
        }

        /// <summary>
        /// Removes an IPv6 address from this VIF
        /// First published in XenServer 6.1.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_value">The IP address which will be removed from the VIF</param>
        public static XenRef<Task> async_remove_ipv6_allowed(Session session, string _vif, string _value)
        {
          return session.JsonRpcClient.async_vif_remove_ipv6_allowed(session.opaque_ref, _vif, _value);
        }

        /// <summary>
        /// Configure IPv4 settings for this virtual interface
        /// First published in XenServer 7.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_mode">Whether to use static or no IPv4 assignment</param>
        /// <param name="_address">The IPv4 address in &lt;addr&gt;/&lt;prefix length&gt; format (for static mode only)</param>
        /// <param name="_gateway">The IPv4 gateway (for static mode only; leave empty to not set a gateway)</param>
        public static void configure_ipv4(Session session, string _vif, vif_ipv4_configuration_mode _mode, string _address, string _gateway)
        {
            session.JsonRpcClient.vif_configure_ipv4(session.opaque_ref, _vif, _mode, _address, _gateway);
        }

        /// <summary>
        /// Configure IPv4 settings for this virtual interface
        /// First published in XenServer 7.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_mode">Whether to use static or no IPv4 assignment</param>
        /// <param name="_address">The IPv4 address in &lt;addr&gt;/&lt;prefix length&gt; format (for static mode only)</param>
        /// <param name="_gateway">The IPv4 gateway (for static mode only; leave empty to not set a gateway)</param>
        public static XenRef<Task> async_configure_ipv4(Session session, string _vif, vif_ipv4_configuration_mode _mode, string _address, string _gateway)
        {
          return session.JsonRpcClient.async_vif_configure_ipv4(session.opaque_ref, _vif, _mode, _address, _gateway);
        }

        /// <summary>
        /// Configure IPv6 settings for this virtual interface
        /// First published in XenServer 7.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_mode">Whether to use static or no IPv6 assignment</param>
        /// <param name="_address">The IPv6 address in &lt;addr&gt;/&lt;prefix length&gt; format (for static mode only)</param>
        /// <param name="_gateway">The IPv6 gateway (for static mode only; leave empty to not set a gateway)</param>
        public static void configure_ipv6(Session session, string _vif, vif_ipv6_configuration_mode _mode, string _address, string _gateway)
        {
            session.JsonRpcClient.vif_configure_ipv6(session.opaque_ref, _vif, _mode, _address, _gateway);
        }

        /// <summary>
        /// Configure IPv6 settings for this virtual interface
        /// First published in XenServer 7.0.
        /// </summary>
        /// <param name="session">The session</param>
        /// <param name="_vif">The opaque_ref of the given vif</param>
        /// <param name="_mode">Whether to use static or no IPv6 assignment</param>
        /// <param name="_address">The IPv6 address in &lt;addr&gt;/&lt;prefix length&gt; format (for static mode only)</param>
        /// <param name="_gateway">The IPv6 gateway (for static mode only; leave empty to not set a gateway)</param>
        public static XenRef<Task> async_configure_ipv6(Session session, string _vif, vif_ipv6_configuration_mode _mode, string _address, string _gateway)
        {
          return session.JsonRpcClient.async_vif_configure_ipv6(session.opaque_ref, _vif, _mode, _address, _gateway);
        }

        /// <summary>
        /// Return a list of all the VIFs known to the system.
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        public static List<XenRef<VIF>> get_all(Session session)
        {
            return session.JsonRpcClient.vif_get_all(session.opaque_ref);
        }

        /// <summary>
        /// Get all the VIF Records at once, in a single XML RPC call
        /// First published in XenServer 4.0.
        /// </summary>
        /// <param name="session">The session</param>
        public static Dictionary<XenRef<VIF>, VIF> get_all_records(Session session)
        {
            return session.JsonRpcClient.vif_get_all_records(session.opaque_ref);
        }

        /// <summary>
        /// Unique identifier/object reference
        /// </summary>
        public virtual string uuid
        {
            get { return _uuid; }
            set
            {
                if (!Helper.AreEqual(value, _uuid))
                {
                    _uuid = value;
                    NotifyPropertyChanged("uuid");
                }
            }
        }
        private string _uuid = "";

        /// <summary>
        /// list of the operations allowed in this state. This list is advisory only and the server state may have changed by the time this field is read by a client.
        /// </summary>
        public virtual List<vif_operations> allowed_operations
        {
            get { return _allowed_operations; }
            set
            {
                if (!Helper.AreEqual(value, _allowed_operations))
                {
                    _allowed_operations = value;
                    NotifyPropertyChanged("allowed_operations");
                }
            }
        }
        private List<vif_operations> _allowed_operations = new List<vif_operations>() {};

        /// <summary>
        /// links each of the running tasks using this object (by reference) to a current_operation enum which describes the nature of the task.
        /// </summary>
        public virtual Dictionary<string, vif_operations> current_operations
        {
            get { return _current_operations; }
            set
            {
                if (!Helper.AreEqual(value, _current_operations))
                {
                    _current_operations = value;
                    NotifyPropertyChanged("current_operations");
                }
            }
        }
        private Dictionary<string, vif_operations> _current_operations = new Dictionary<string, vif_operations>() {};

        /// <summary>
        /// order in which VIF backends are created by xapi
        /// </summary>
        public virtual string device
        {
            get { return _device; }
            set
            {
                if (!Helper.AreEqual(value, _device))
                {
                    _device = value;
                    NotifyPropertyChanged("device");
                }
            }
        }
        private string _device = "";

        /// <summary>
        /// virtual network to which this vif is connected
        /// </summary>
        [JsonConverter(typeof(XenRefConverter<Network>))]
        public virtual XenRef<Network> network
        {
            get { return _network; }
            set
            {
                if (!Helper.AreEqual(value, _network))
                {
                    _network = value;
                    NotifyPropertyChanged("network");
                }
            }
        }
        private XenRef<Network> _network = new XenRef<Network>(Helper.NullOpaqueRef);

        /// <summary>
        /// virtual machine to which this vif is connected
        /// </summary>
        [JsonConverter(typeof(XenRefConverter<VM>))]
        public virtual XenRef<VM> VM
        {
            get { return _VM; }
            set
            {
                if (!Helper.AreEqual(value, _VM))
                {
                    _VM = value;
                    NotifyPropertyChanged("VM");
                }
            }
        }
        private XenRef<VM> _VM = new XenRef<VM>(Helper.NullOpaqueRef);

        /// <summary>
        /// ethernet MAC address of virtual interface, as exposed to guest
        /// </summary>
        public virtual string MAC
        {
            get { return _MAC; }
            set
            {
                if (!Helper.AreEqual(value, _MAC))
                {
                    _MAC = value;
                    NotifyPropertyChanged("MAC");
                }
            }
        }
        private string _MAC = "";

        /// <summary>
        /// MTU in octets
        /// </summary>
        public virtual long MTU
        {
            get { return _MTU; }
            set
            {
                if (!Helper.AreEqual(value, _MTU))
                {
                    _MTU = value;
                    NotifyPropertyChanged("MTU");
                }
            }
        }
        private long _MTU;

        /// <summary>
        /// additional configuration
        /// </summary>
        [JsonConverter(typeof(StringStringMapConverter))]
        public virtual Dictionary<string, string> other_config
        {
            get { return _other_config; }
            set
            {
                if (!Helper.AreEqual(value, _other_config))
                {
                    _other_config = value;
                    NotifyPropertyChanged("other_config");
                }
            }
        }
        private Dictionary<string, string> _other_config = new Dictionary<string, string>() {};

        /// <summary>
        /// is the device currently attached (erased on reboot)
        /// </summary>
        public virtual bool currently_attached
        {
            get { return _currently_attached; }
            set
            {
                if (!Helper.AreEqual(value, _currently_attached))
                {
                    _currently_attached = value;
                    NotifyPropertyChanged("currently_attached");
                }
            }
        }
        private bool _currently_attached = false;

        /// <summary>
        /// error/success code associated with last attach-operation (erased on reboot)
        /// </summary>
        public virtual long status_code
        {
            get { return _status_code; }
            set
            {
                if (!Helper.AreEqual(value, _status_code))
                {
                    _status_code = value;
                    NotifyPropertyChanged("status_code");
                }
            }
        }
        private long _status_code;

        /// <summary>
        /// error/success information associated with last attach-operation status (erased on reboot)
        /// </summary>
        public virtual string status_detail
        {
            get { return _status_detail; }
            set
            {
                if (!Helper.AreEqual(value, _status_detail))
                {
                    _status_detail = value;
                    NotifyPropertyChanged("status_detail");
                }
            }
        }
        private string _status_detail = "";

        /// <summary>
        /// Device runtime properties
        /// </summary>
        [JsonConverter(typeof(StringStringMapConverter))]
        public virtual Dictionary<string, string> runtime_properties
        {
            get { return _runtime_properties; }
            set
            {
                if (!Helper.AreEqual(value, _runtime_properties))
                {
                    _runtime_properties = value;
                    NotifyPropertyChanged("runtime_properties");
                }
            }
        }
        private Dictionary<string, string> _runtime_properties = new Dictionary<string, string>() {};

        /// <summary>
        /// QoS algorithm to use
        /// </summary>
        public virtual string qos_algorithm_type
        {
            get { return _qos_algorithm_type; }
            set
            {
                if (!Helper.AreEqual(value, _qos_algorithm_type))
                {
                    _qos_algorithm_type = value;
                    NotifyPropertyChanged("qos_algorithm_type");
                }
            }
        }
        private string _qos_algorithm_type = "";

        /// <summary>
        /// parameters for chosen QoS algorithm
        /// </summary>
        [JsonConverter(typeof(StringStringMapConverter))]
        public virtual Dictionary<string, string> qos_algorithm_params
        {
            get { return _qos_algorithm_params; }
            set
            {
                if (!Helper.AreEqual(value, _qos_algorithm_params))
                {
                    _qos_algorithm_params = value;
                    NotifyPropertyChanged("qos_algorithm_params");
                }
            }
        }
        private Dictionary<string, string> _qos_algorithm_params = new Dictionary<string, string>() {};

        /// <summary>
        /// supported QoS algorithms for this VIF
        /// </summary>
        public virtual string[] qos_supported_algorithms
        {
            get { return _qos_supported_algorithms; }
            set
            {
                if (!Helper.AreEqual(value, _qos_supported_algorithms))
                {
                    _qos_supported_algorithms = value;
                    NotifyPropertyChanged("qos_supported_algorithms");
                }
            }
        }
        private string[] _qos_supported_algorithms = {};

        /// <summary>
        /// metrics associated with this VIF
        /// </summary>
        [JsonConverter(typeof(XenRefConverter<VIF_metrics>))]
        public virtual XenRef<VIF_metrics> metrics
        {
            get { return _metrics; }
            set
            {
                if (!Helper.AreEqual(value, _metrics))
                {
                    _metrics = value;
                    NotifyPropertyChanged("metrics");
                }
            }
        }
        private XenRef<VIF_metrics> _metrics = new XenRef<VIF_metrics>("OpaqueRef:NULL");

        /// <summary>
        /// true if the MAC was autogenerated; false indicates it was set manually
        /// First published in XenServer 5.5.
        /// </summary>
        public virtual bool MAC_autogenerated
        {
            get { return _MAC_autogenerated; }
            set
            {
                if (!Helper.AreEqual(value, _MAC_autogenerated))
                {
                    _MAC_autogenerated = value;
                    NotifyPropertyChanged("MAC_autogenerated");
                }
            }
        }
        private bool _MAC_autogenerated = false;

        /// <summary>
        /// current locking mode of the VIF
        /// First published in XenServer 6.1.
        /// </summary>
        [JsonConverter(typeof(vif_locking_modeConverter))]
        public virtual vif_locking_mode locking_mode
        {
            get { return _locking_mode; }
            set
            {
                if (!Helper.AreEqual(value, _locking_mode))
                {
                    _locking_mode = value;
                    NotifyPropertyChanged("locking_mode");
                }
            }
        }
        private vif_locking_mode _locking_mode = vif_locking_mode.network_default;

        /// <summary>
        /// A list of IPv4 addresses which can be used to filter traffic passing through this VIF
        /// First published in XenServer 6.1.
        /// </summary>
        public virtual string[] ipv4_allowed
        {
            get { return _ipv4_allowed; }
            set
            {
                if (!Helper.AreEqual(value, _ipv4_allowed))
                {
                    _ipv4_allowed = value;
                    NotifyPropertyChanged("ipv4_allowed");
                }
            }
        }
        private string[] _ipv4_allowed = {};

        /// <summary>
        /// A list of IPv6 addresses which can be used to filter traffic passing through this VIF
        /// First published in XenServer 6.1.
        /// </summary>
        public virtual string[] ipv6_allowed
        {
            get { return _ipv6_allowed; }
            set
            {
                if (!Helper.AreEqual(value, _ipv6_allowed))
                {
                    _ipv6_allowed = value;
                    NotifyPropertyChanged("ipv6_allowed");
                }
            }
        }
        private string[] _ipv6_allowed = {};

        /// <summary>
        /// Determines whether IPv4 addresses are configured on the VIF
        /// First published in XenServer 7.0.
        /// </summary>
        [JsonConverter(typeof(vif_ipv4_configuration_modeConverter))]
        public virtual vif_ipv4_configuration_mode ipv4_configuration_mode
        {
            get { return _ipv4_configuration_mode; }
            set
            {
                if (!Helper.AreEqual(value, _ipv4_configuration_mode))
                {
                    _ipv4_configuration_mode = value;
                    NotifyPropertyChanged("ipv4_configuration_mode");
                }
            }
        }
        private vif_ipv4_configuration_mode _ipv4_configuration_mode = vif_ipv4_configuration_mode.None;

        /// <summary>
        /// IPv4 addresses in CIDR format
        /// First published in XenServer 7.0.
        /// </summary>
        public virtual string[] ipv4_addresses
        {
            get { return _ipv4_addresses; }
            set
            {
                if (!Helper.AreEqual(value, _ipv4_addresses))
                {
                    _ipv4_addresses = value;
                    NotifyPropertyChanged("ipv4_addresses");
                }
            }
        }
        private string[] _ipv4_addresses = {};

        /// <summary>
        /// IPv4 gateway (the empty string means that no gateway is set)
        /// First published in XenServer 7.0.
        /// </summary>
        public virtual string ipv4_gateway
        {
            get { return _ipv4_gateway; }
            set
            {
                if (!Helper.AreEqual(value, _ipv4_gateway))
                {
                    _ipv4_gateway = value;
                    NotifyPropertyChanged("ipv4_gateway");
                }
            }
        }
        private string _ipv4_gateway = "";

        /// <summary>
        /// Determines whether IPv6 addresses are configured on the VIF
        /// First published in XenServer 7.0.
        /// </summary>
        [JsonConverter(typeof(vif_ipv6_configuration_modeConverter))]
        public virtual vif_ipv6_configuration_mode ipv6_configuration_mode
        {
            get { return _ipv6_configuration_mode; }
            set
            {
                if (!Helper.AreEqual(value, _ipv6_configuration_mode))
                {
                    _ipv6_configuration_mode = value;
                    NotifyPropertyChanged("ipv6_configuration_mode");
                }
            }
        }
        private vif_ipv6_configuration_mode _ipv6_configuration_mode = vif_ipv6_configuration_mode.None;

        /// <summary>
        /// IPv6 addresses in CIDR format
        /// First published in XenServer 7.0.
        /// </summary>
        public virtual string[] ipv6_addresses
        {
            get { return _ipv6_addresses; }
            set
            {
                if (!Helper.AreEqual(value, _ipv6_addresses))
                {
                    _ipv6_addresses = value;
                    NotifyPropertyChanged("ipv6_addresses");
                }
            }
        }
        private string[] _ipv6_addresses = {};

        /// <summary>
        /// IPv6 gateway (the empty string means that no gateway is set)
        /// First published in XenServer 7.0.
        /// </summary>
        public virtual string ipv6_gateway
        {
            get { return _ipv6_gateway; }
            set
            {
                if (!Helper.AreEqual(value, _ipv6_gateway))
                {
                    _ipv6_gateway = value;
                    NotifyPropertyChanged("ipv6_gateway");
                }
            }
        }
        private string _ipv6_gateway = "";
    }
}
