﻿/* Copyright (c) Cloud Software Group, Inc. 
 * 
 * Redistribution and use in source and binary forms, 
 * with or without modification, are permitted provided 
 * that the following conditions are met: 
 * 
 * *   Redistributions of source code must retain the above 
 *     copyright notice, this list of conditions and the 
 *     following disclaimer. 
 * *   Redistributions in binary form must reproduce the above 
 *     copyright notice, this list of conditions and the 
 *     following disclaimer in the documentation and/or other 
 *     materials provided with the distribution. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND 
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 */

using System;
using System.IO;
using System.IO.Compression;


namespace XenCenterLib.Archive
{
    internal class ZipArchiveWriter : ArchiveWriter
    {
        private ZipArchive _zipArchive;
        private bool _disposed;

        /// <summary>
        /// Parameterless constructor needed by tests
        /// </summary>
        public ZipArchiveWriter()
        {
        }

        public ZipArchiveWriter(Stream outputStream)
        {
            _zipArchive = new ZipArchive(outputStream, ZipArchiveMode.Create);
        }

        public override void SetBaseStream(Stream outputStream)
        {
            _zipArchive = new ZipArchive(outputStream, ZipArchiveMode.Create);
            _disposed = false;
        }

        public override void AddDirectory(string directoryName, DateTime modificationTime)
        {
            //Need to add a trailing front-slash to add a directory
            if (!directoryName.EndsWith("/"))
                directoryName += "/";

            var entry = _zipArchive.CreateEntry(directoryName);
            entry.LastWriteTime = modificationTime;
        }

        public override void Add(Stream fileToAdd, string fileName, DateTime modificationTime, Action cancellingDelegate)
        {
            var entry = _zipArchive.CreateEntry(fileName);
            entry.LastWriteTime = modificationTime;

            //You have to do this because if using a memory stream the pointer will be at the end it
            //it's just been read and this will cause nothing to be written out
            fileToAdd.Position = 0;

            using (var entryStream = entry.Open())
                StreamUtilities.BufferedStreamCopy(fileToAdd, entryStream);
        }

        protected override void Dispose(bool disposing)
        {
            if (disposing && !_disposed)
            {
                _zipArchive?.Dispose();
                _disposed = true;
            }

            base.Dispose(disposing);
        }
    }
}
