﻿/* Copyright (c) Cloud Software Group, Inc. 
 * 
 * Redistribution and use in source and binary forms, 
 * with or without modification, are permitted provided 
 * that the following conditions are met: 
 * 
 * *   Redistributions of source code must retain the above 
 *     copyright notice, this list of conditions and the 
 *     following disclaimer. 
 * *   Redistributions in binary form must reproduce the above 
 *     copyright notice, this list of conditions and the 
 *     following disclaimer in the documentation and/or other 
 *     materials provided with the distribution. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND 
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 */

using System;
using System.Collections.Generic;
using XenAdmin.Diagnostics.Hotfixing;
using XenAdmin.Diagnostics.Problems;
using XenAdmin.Diagnostics.Problems.HostProblem;
using XenAPI;


namespace XenAdmin.Diagnostics.Checks
{
    class PrepareToUpgradeCheck : HostPostLivenessCheck
    {
        private static readonly log4net.ILog log = log4net.LogManager.GetLogger(System.Reflection.MethodBase.GetCurrentMethod().DeclaringType);

        private readonly Dictionary<string, string> _installMethodConfig;

        public PrepareToUpgradeCheck(Host host, Dictionary<string, string> installMethodConfig)
            : base(host)
        {
            _installMethodConfig = installMethodConfig;
        }

        public override string Description => Messages.CHECKING_PREPARE_TO_UPGRADE_DESCRIPTION;

        protected override Problem RunHostCheck()
        {
            var hotfix = HotfixFactory.Hotfix(Host);
            if (hotfix != null && hotfix.ShouldBeAppliedTo(Host))
                return new HostDoesNotHaveHotfixWarning(this, Host);

            try
            {
                var result = Host.call_plugin(Host.Connection.Session, Host.opaque_ref,
                    "prepare_host_upgrade.py", "testUrl", _installMethodConfig);

                if (result.ToLower() == "true")
                    return null;
            }
            catch (Exception exception)
            {
                var failure = exception as Failure;
                if (failure?.ErrorDescription.Count == 4)
                    return new HostPrepareToUpgradeProblem(this, Host, failure.ErrorDescription[3]);

                log.Error("Error testing upgrade hotfix.", exception);
            }

            return new HostPrepareToUpgradeProblem(this, Host);
        }
    }
}
